<?php
/**
 * SentraSystems Plugin Auto-Updater
 *
 * Checks sentrasys.dev/api/sentra-repo for new versions and lets WordPress
 * update this plugin automatically through the standard update mechanism.
 *
 * Endpoint used:
 *   GET /api/sentra-repo/public/latest/{app_id}/{channel}
 *   GET /api/sentra-repo/download/{version_id}   (served by sentra-core)
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class SentraSystems_Plugin_Updater {

    /** App ID registered in the Sentra Repo for the plugin */
    const APP_ID  = 'sentra-connector';

    /** Release channel to track  */
    const CHANNEL = 'stable';

    /** Manifest endpoint base URL — no trailing slash */
    const REPO_BASE = 'https://sentrasys.dev/api/sentra-repo';

    /** WordPress plugin slug: folder/main-file.php */
    const PLUGIN_SLUG = 'sentra-connector/sentrasystems.php';

    /** How long (seconds) to cache the remote version check */
    const CACHE_TTL = 43200; // 12 hours

    /** Transient key for cached manifest data */
    const TRANSIENT_KEY = 'sentrasystems_update_manifest';

    public function __construct() {
        add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'inject_update' ] );
        add_filter( 'plugins_api',                           [ $this, 'plugin_info' ], 10, 3 );
        add_action( 'upgrader_process_complete',             [ $this, 'purge_cache' ],  10, 2 );
    }

    // -------------------------------------------------------------------------
    // Fetch & cache the remote manifest
    // -------------------------------------------------------------------------

    private function get_manifest(): ?array {
        $cached = get_site_transient( self::TRANSIENT_KEY );
        if ( is_array( $cached ) && ! empty( $cached['version'] ) ) {
            return $cached;
        }

        $url = self::REPO_BASE . '/public/latest/' . self::APP_ID . '/' . self::CHANNEL;

        $response = wp_remote_get( $url, [
            'timeout'    => 10,
            'user-agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; SentraSystems-Updater/' . SENTRASYSTEMS_VERSION,
            'headers'    => [
                'Accept' => 'application/json',
            ],
        ] );

        if ( is_wp_error( $response ) ) {
            return null;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code !== 200 ) {
            return null;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! is_array( $data ) || empty( $data['version'] ) ) {
            return null;
        }

        // Normalise download URL: if relative, make it absolute
        if ( ! empty( $data['download_url'] ) && strpos( $data['download_url'], 'http' ) !== 0 ) {
            $data['download_url'] = 'https://sentrasys.dev' . $data['download_url'];
        }

        set_site_transient( self::TRANSIENT_KEY, $data, self::CACHE_TTL );

        return $data;
    }

    // -------------------------------------------------------------------------
    // Hook: inject into WordPress update transient
    // -------------------------------------------------------------------------

    public function inject_update( $transient ) {
        if ( empty( $transient->checked ) ) {
            return $transient;
        }

        $manifest = $this->get_manifest();
        if ( ! $manifest ) {
            return $transient;
        }

        $remote_version  = $manifest['version'];
        $current_version = $transient->checked[ self::PLUGIN_SLUG ] ?? SENTRASYSTEMS_VERSION;

        if ( version_compare( $remote_version, $current_version, '>' ) ) {
            $transient->response[ self::PLUGIN_SLUG ] = (object) [
                'id'            => 'sentra-repo/' . self::APP_ID,
                'slug'          => dirname( self::PLUGIN_SLUG ),
                'plugin'        => self::PLUGIN_SLUG,
                'new_version'   => $remote_version,
                'url'           => $manifest['homepage'] ?? 'https://sentrasys.dev',
                'package'       => $manifest['download_url'],
                'icons'         => [],
                'banners'       => [],
                'banners_rtl'   => [],
                'tested'        => $manifest['tested']   ?? '',
                'requires_php'  => $manifest['requires_php'] ?? '7.4',
                'requires'      => $manifest['requires']  ?? '6.1',
                'upgrade_notice'=> $manifest['notes']     ?? '',
            ];
        } else {
            // Explicitly mark as up-to-date so WP doesn't think it's unknown
            $transient->no_update[ self::PLUGIN_SLUG ] = (object) [
                'id'          => 'sentra-repo/' . self::APP_ID,
                'slug'        => dirname( self::PLUGIN_SLUG ),
                'plugin'      => self::PLUGIN_SLUG,
                'new_version' => $current_version,
                'url'         => $manifest['homepage'] ?? 'https://sentrasys.dev',
                'package'     => '',
            ];
        }

        return $transient;
    }

    // -------------------------------------------------------------------------
    // Hook: supply plugin info for the "View details" pop-up
    // -------------------------------------------------------------------------

    public function plugin_info( $result, string $action, $args ) {
        if ( $action !== 'plugin_information' ) {
            return $result;
        }

        if ( ! isset( $args->slug ) || $args->slug !== dirname( self::PLUGIN_SLUG ) ) {
            return $result;
        }

        $manifest = $this->get_manifest();
        if ( ! $manifest ) {
            return $result;
        }

        $changelog = $manifest['changelog'] ?? $manifest['notes'] ?? '';
        $sections  = [
            'description' => 'Sentra Systems WordPress Connector — integrates your site with the Sentra platform.',
            'changelog'   => $changelog ?: 'See <a href="https://sentrasys.dev">sentrasys.dev</a> for release notes.',
        ];

        return (object) [
            'name'          => 'SentraSystems',
            'slug'          => dirname( self::PLUGIN_SLUG ),
            'version'       => $manifest['version'],
            'author'        => '<a href="https://sentrasys.dev">Sentra Systems</a>',
            'homepage'      => $manifest['homepage'] ?? 'https://sentrasys.dev',
            'requires'      => $manifest['requires']     ?? '6.1',
            'tested'        => $manifest['tested']       ?? '',
            'requires_php'  => $manifest['requires_php'] ?? '7.4',
            'download_link' => $manifest['download_url'],
            'last_updated'  => $manifest['last_updated'] ?? '',
            'sections'      => $sections,
        ];
    }

    // -------------------------------------------------------------------------
    // Hook: purge cache after an upgrade so next check is fresh
    // -------------------------------------------------------------------------

    public function purge_cache( $upgrader, array $hook_extra ): void {
        if ( isset( $hook_extra['plugin'] ) && $hook_extra['plugin'] === self::PLUGIN_SLUG ) {
            delete_site_transient( self::TRANSIENT_KEY );
        }
    }
}
